<?php
declare(strict_types=1);

require_once __DIR__ . '/../utils.php';
require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../auth.php';

$admin = require_role('admin');

$body = get_request_json();
$fullName = safe_string($body['full_name'] ?? '', 150);
$phone    = safe_string($body['phone'] ?? '', 30);
$email    = safe_string($body['email'] ?? '', 150);

// Optional: allow admin to provide temp password; else generate
$tempPassword = (string)($body['temp_password'] ?? '');

if ($fullName === '' || $phone === '' || $email === '') {
  json_response(['ok' => false, 'error' => 'VALIDATION', 'message' => 'full_name, phone, email are required'], 422);
}

if ($tempPassword === '') {
  // simple generated temp password
  $tempPassword = 'Owner@' . substr(bin2hex(random_bytes(4)), 0, 6);
}

$hash = password_hash($tempPassword, PASSWORD_DEFAULT);

try {
  // avoid duplicates
  $chk = db()->prepare("SELECT id FROM users WHERE email = ? OR phone = ? LIMIT 1");
  $chk->execute([$email, $phone]);
  if ($chk->fetch()) {
    json_response(['ok' => false, 'error' => 'DUPLICATE', 'message' => 'Owner with email or phone already exists'], 409);
  }

  $stmt = db()->prepare("
    INSERT INTO users (role, full_name, phone, email, password_hash, must_change_password, is_active, created_at)
    VALUES ('owner', ?, ?, ?, ?, 1, 1, ?)
  ");
  $stmt->execute([$fullName, $phone, $email, $hash, now_utc()]);

  $ownerId = (int)db()->lastInsertId();

  json_response([
    'ok' => true,
    'owner' => [
      'id' => $ownerId,
      'full_name' => $fullName,
      'phone' => $phone,
      'email' => $email
    ],
    'temp_password' => $tempPassword
  ]);
} catch (Throwable $e) {
  json_response(['ok' => false, 'error' => 'CREATE_FAILED', 'message' => APP_DEBUG ? $e->getMessage() : 'Failed'], 500);
}
