<?php
declare(strict_types=1);

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/utils.php';

/**
 * Safe router (no syntax shortcuts).
 * Routes /shop-system/api/<path> to corresponding PHP handler.
 */

// Always return JSON on fatal error (dev-friendly)
register_shutdown_function(function () {
  $err = error_get_last();
  if ($err && in_array($err['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR], true)) {
    http_response_code(500);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
      'ok' => false,
      'error' => 'FATAL_ERROR',
      'message' => (defined('APP_DEBUG') && APP_DEBUG)
        ? ($err['message'] . ' at ' . $err['file'] . ':' . $err['line'])
        : 'Server error'
    ]);
  }
});

$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');

// Determine path
$path = $_SERVER['PATH_INFO'] ?? null;

if (!$path) {
  $uriPath = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
  if (!is_string($uriPath)) $uriPath = '/';

  $scriptName = str_replace('\\', '/', $_SERVER['SCRIPT_NAME'] ?? '');
  $baseDir = rtrim(str_replace('\\', '/', dirname($scriptName)), '/'); // e.g. /shop-system/api

  if ($baseDir !== '' && str_starts_with($uriPath, $baseDir)) {
    $uriPath = substr($uriPath, strlen($baseDir));
  }

  // remove /index.php if present
  $uriPath = preg_replace('#^/index\.php#', '', $uriPath);

  $path = '/' . ltrim($uriPath, '/');
}

if ($path === '') $path = '/';

// ROUTES
$routes = [
  'GET' => [
    '/public/ping' => __DIR__ . '/public/ping.php',
    '/admin/dashboard_stats' => __DIR__ . '/admin/dashboard_stats.php',
    '/admin/reports_list' => __DIR__ . '/admin/reports_list.php',
    '/admin/list_shops' => __DIR__ . '/admin/list_shops.php',
    '/admin/owner_details' => __DIR__ . '/admin/owner_details.php',
    '/admin/shop_details' => __DIR__ . '/admin/shop_details.php',
    '/owner/my_shop' => __DIR__ . '/owner/my_shop.php',
    '/owner/dashboard_stats' => __DIR__ . '/owner/dashboard_stats.php',
'/owner/list_products' => __DIR__ . '/owner/list_products.php',
'/public/list_shops' => __DIR__ . '/public/list_shops.php',
'/public/shop_details' => __DIR__ . '/public/shop_details.php',
'/public/list_products' => __DIR__ . '/public/list_products.php',
'/public/product_details' => __DIR__ . '/public/product_details.php',
'/owner/list_orders' => __DIR__ . '/owner/list_orders.php',
    '/public/reviews_for_shop' => __DIR__ . '/public/reviews_for_shop.php',
    '/admin/list_owners' => __DIR__ . '/admin/list_owners.php',
    '/admin/list_owners_all' => __DIR__ . '/admin/list_owners_all.php',





  ],
  'POST' => [
    '/auth/login' => __DIR__ . '/auth/login.php',
    '/auth/logout' => __DIR__ . '/auth/logout.php',
    '/auth/change_password' => __DIR__ . '/auth/change_password.php',

    '/admin/create_shop_owner' => __DIR__ . '/admin/create_shop_owner.php',
    '/admin/create_shop'       => __DIR__ . '/admin/create_shop.php',
    '/admin/verify_shop'       => __DIR__ . '/admin/verify_shop.php',
    '/admin/suspend_shop'      => __DIR__ . '/admin/suspend_shop.php',
    '/admin/activate_owner'    => __DIR__ . '/admin/activate_owner.php',
    '/admin/reset_owner_password' => __DIR__ . '/admin/reset_owner_password.php',
    '/owner/update_shop' => __DIR__ . '/owner/update_shop.php',
'/owner/create_product' => __DIR__ . '/owner/create_product.php',
'/owner/update_product' => __DIR__ . '/owner/update_product.php',
'/owner/delete_product' => __DIR__ . '/owner/delete_product.php',
'/public/create_order' => __DIR__ . '/public/create_order.php',
'/owner/complete_order' => __DIR__ . '/owner/complete_order.php',
'/owner/create_review_token' => __DIR__ . '/owner/create_review_token.php',
'/public/review_submit' => __DIR__ . '/public/review_submit.php',
'/owner/update_profile' => __DIR__ . '/owner/update_profile.php',




  ],
];

// DISPATCH
if (isset($routes[$method]) && isset($routes[$method][$path])) {
  $handler = $routes[$method][$path];
  if (!file_exists($handler)) {
    json_response([
      'ok' => false,
      'error' => 'HANDLER_MISSING',
      'handler' => $handler,
      'method' => $method,
      'path' => $path
    ], 500);
  }
  require $handler;
  exit;
}

json_response([
  'ok' => false,
  'error' => 'NOT_FOUND',
  'path' => $path,
  'method' => $method
], 404);
