<?php
declare(strict_types=1);

require_once __DIR__ . '/../utils.php';
require_once __DIR__ . '/../db.php';

$body = get_request_json();

$token = (string)($body['token'] ?? '');
$rating = (int)($body['rating'] ?? 0);
$comment = trim((string)($body['comment'] ?? ''));

if ($token === '' || $rating < 1 || $rating > 5) {
  json_response(['ok' => false, 'error' => 'VALIDATION', 'message' => 'token and rating(1-5) required'], 422);
}

$hash = hash('sha256', $token);

db()->beginTransaction();
try {
  // Lock token row to prevent double-submit race
  $stmt = db()->prepare("
    SELECT id, shop_id, order_intent_id, expires_at, used_at
    FROM review_tokens
    WHERE token_hash=?
    LIMIT 1
    FOR UPDATE
  ");
  $stmt->execute([$hash]);
  $t = $stmt->fetch();
  if (!$t) {
    db()->rollBack();
    json_response(['ok' => false, 'error' => 'TOKEN_INVALID'], 400);
  }

  if (!empty($t['used_at'])) {
    db()->rollBack();
    json_response(['ok' => false, 'error' => 'TOKEN_USED'], 400);
  }

  $exp = strtotime((string)$t['expires_at']);
  if ($exp !== false && $exp < time()) {
    db()->rollBack();
    json_response(['ok' => false, 'error' => 'TOKEN_EXPIRED'], 400);
  }

  $ins = db()->prepare("
    INSERT INTO reviews (shop_id, order_intent_id, rating, comment, created_at)
    VALUES (?, ?, ?, ?, ?)
  ");
  $ins->execute([
    (int)$t['shop_id'],
    (int)$t['order_intent_id'],
    $rating,
    $comment !== '' ? $comment : null,
    now_utc()
  ]);

  $upd = db()->prepare("UPDATE review_tokens SET used_at=? WHERE id=?");
  $upd->execute([now_utc(), (int)$t['id']]);

  db()->commit();
  json_response(['ok' => true]);
} catch (Throwable $e) {
  db()->rollBack();
  json_response(['ok' => false, 'error' => 'REVIEW_SUBMIT_FAILED', 'message' => $e->getMessage()], 400);
}
