const CART_KEY = "daura_cart_v1";

export function getCart() {
  try { return JSON.parse(localStorage.getItem(CART_KEY)) || null; }
  catch { return null; }
}

export function clearCart() {
  localStorage.removeItem(CART_KEY);
}

export function addToCart(shop, product, qty = 1) {
  const cart = getCart();

  const item = {
    product_id: Number(product.id),
    name: product.name,
    price: Number(product.price),
    qty: Math.max(1, Number(qty) || 1),
    image_path: product.image_path || null,
  };

  // One-shop rule
  if (cart && cart.shop_id !== Number(shop.id)) {
    return { ok: false, error: "ONE_SHOP_ONLY", message: "Cart already has items from another shop. Clear cart first." };
  }

  const next = cart || {
    shop_id: Number(shop.id),
    shop_name: shop.name,
    whatsapp_phone: shop.whatsapp_phone,
    items: []
  };

  const existing = next.items.find(i => i.product_id === item.product_id);
  if (existing) existing.qty += item.qty;
  else next.items.push(item);

  localStorage.setItem(CART_KEY, JSON.stringify(next));
  return { ok: true, cart: next };
}

export function updateQty(product_id, qty) {
  const cart = getCart();
  if (!cart) return { ok: false };
  const it = cart.items.find(i => i.product_id === Number(product_id));
  if (!it) return { ok: false };
  it.qty = Math.max(1, Number(qty) || 1);
  localStorage.setItem(CART_KEY, JSON.stringify(cart));
  return { ok: true, cart };
}

export function removeItem(product_id) {
  const cart = getCart();
  if (!cart) return { ok: false };
  cart.items = cart.items.filter(i => i.product_id !== Number(product_id));
  if (cart.items.length === 0) clearCart();
  else localStorage.setItem(CART_KEY, JSON.stringify(cart));
  return { ok: true, cart: getCart() };
}

export function cartCount(cart) {
  if (!cart) return 0;
  return cart.items.reduce((n, i) => n + Number(i.qty || 0), 0);
}

export function cartTotal(cart) {
  if (!cart) return 0;
  return cart.items.reduce((sum, i) => sum + (Number(i.price) * Number(i.qty)), 0);
}
