import { API_BASE } from "./config.js";
import { authHeaders, getUser, logout, requireRole, setUser } from "./auth.js";

const ownerName = document.getElementById("owner-name");
const ownerAvatar = document.getElementById("owner-avatar");
const logoutBtn = document.getElementById("owner-logout");
const tabs = document.getElementById("owner-tabs");
const shopBox = document.getElementById("owner-shop");
const shopForm = document.getElementById("update-shop-form");
const shopResult = document.getElementById("shop-update-result");
const profileForm = document.getElementById("update-profile-form");
const profileResult = document.getElementById("profile-update-result");
const passwordForm = document.getElementById("change-password-form");
const passwordResult = document.getElementById("password-change-result");
const passwordNote = document.getElementById("owner-password-note");
const productsBox = document.getElementById("owner-products");
const createProductForm = document.getElementById("create-product-form");
const updateProductForm = document.getElementById("update-product-form");
const createProductResult = document.getElementById("product-create-result");
const updateProductResult = document.getElementById("product-update-result");
const updateProductSelect = document.getElementById("update-product-id");
const updateProductName = document.getElementById("update-product-name");
const updateProductDescription = document.getElementById("update-product-description");
const updateProductPrice = document.getElementById("update-product-price");
const updateProductActive = document.getElementById("update-product-active");
const ordersBox = document.getElementById("owner-orders");
const orderMessage = document.getElementById("order-message");
const openSecurityBtn = document.getElementById("open-security");
const statOrders = document.getElementById("owner-stat-orders");
const statCompleted = document.getElementById("owner-stat-completed");
const statReviews = document.getElementById("owner-stat-reviews");

let productsCache = [];
let mustChangePassword = false;

const user = requireRole("owner");
if (!user) {
  throw new Error("Unauthorized");
}
ownerName.textContent = user.full_name || user.email || "Owner";
if (ownerAvatar) {
  const label = ownerName.textContent || "O";
  ownerAvatar.textContent = label.trim().slice(0, 1).toUpperCase();
}
mustChangePassword = Number(user.must_change_password || 0) === 1;
if (mustChangePassword && passwordNote) {
  passwordNote.textContent = "Please change your temporary password to activate your shop.";
  passwordNote.style.display = "block";
}
if (mustChangePassword) {
  setActiveTab("security");
  [shopForm, profileForm, createProductForm, updateProductForm].forEach((form) => {
    if (!form) return;
    Array.from(form.elements || []).forEach((el) => {
      el.disabled = true;
    });
  });
}

logoutBtn?.addEventListener("click", async () => {
  await logout();
  window.location.href = "/shop-system/app/login.html";
});

tabs?.addEventListener("click", (event) => {
  const btn = event.target.closest(".tab");
  if (!btn) return;
  setActiveTab(btn.dataset.tab);
});

function setActiveTab(tab) {
  if (!tab) return;
  document.querySelectorAll(".tab").forEach((el) => {
    el.classList.toggle("active", el.dataset.tab === tab);
  });
  document.querySelectorAll(".panel").forEach((panel) => {
    panel.classList.toggle("active", panel.dataset.panel === tab);
  });
}

async function apiRequest(path, options = {}) {
  const res = await fetch(`${API_BASE}${path}`, {
    ...options,
    headers: {
      ...authHeaders(),
      ...(options.headers || {})
    }
  });
  const text = await res.text();
  try {
    return JSON.parse(text);
  } catch {
    return { ok: false, error: "BAD_JSON", raw: text };
  }
}

async function apiGet(path) {
  return apiRequest(path, { method: "GET" });
}

async function apiPost(path, payload) {
  return apiRequest(path, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(payload)
  });
}

async function apiPostForm(path, formData) {
  return apiRequest(path, {
    method: "POST",
    body: formData
  });
}

function showNotice(el, text, isError = false) {
  if (!el) return;
  el.textContent = text;
  el.style.display = "block";
  el.style.background = isError ? "#fff1f0" : "#f0fff6";
  el.style.borderColor = isError ? "rgba(159, 77, 47, 0.3)" : "rgba(47, 93, 80, 0.3)";
}

function setStat(el, value, fallback = "--") {
  if (!el) return;
  el.textContent = value === null || value === undefined ? fallback : String(value);
}

async function loadStats() {
  const data = await apiGet("/owner/dashboard_stats");
  if (!data.ok) {
    setStat(statOrders, null);
    setStat(statCompleted, null);
    setStat(statReviews, null);
    return;
  }
  const stats = data.stats || {};
  setStat(statOrders, stats.total_orders ?? 0);
  setStat(statCompleted, stats.completed_orders ?? 0);
  setStat(statReviews, stats.total_reviews ?? 0);
}

function shopCard(shop) {
  return `
    <div>
      <strong>${shop.name}</strong><br />
      <span class="muted">${shop.description || "No description"}</span><br />
      <span class="badge ${Number(shop.is_verified) ? "good" : "warn"}">
        ${Number(shop.is_verified) ? "Verified" : "Unverified"}
      </span>
      <span class="badge ${Number(shop.is_suspended) ? "warn" : "good"}">
        ${Number(shop.is_suspended) ? "Suspended" : "Active"}
      </span>
    </div>
  `;
}

async function loadShop() {
  const data = await apiGet("/owner/my_shop");
  if (!data.ok) {
    shopBox.textContent = data.message || "Unable to load shop details.";
    return;
  }
  shopBox.innerHTML = shopCard(data.shop);
  if (shopForm) {
    shopForm.name.value = data.shop.name || "";
    shopForm.description.value = data.shop.description || "";
    shopForm.whatsapp_phone.value = data.shop.whatsapp_phone || "";
    shopForm.address.value = data.shop.address || "";
  }
}

function fillProfile() {
  const currentUser = getUser();
  if (!currentUser || !profileForm) return;
  profileForm.full_name.value = currentUser.full_name || "";
  profileForm.phone.value = currentUser.phone || "";
  profileForm.email.value = currentUser.email || "";
}

shopForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  shopResult.style.display = "none";
  const formData = new FormData(shopForm);
  const payload = {
    name: String(formData.get("name") || "").trim(),
    description: String(formData.get("description") || "").trim(),
    whatsapp_phone: String(formData.get("whatsapp_phone") || "").trim(),
    address: String(formData.get("address") || "").trim()
  };
  const data = await apiPost("/owner/update_shop", payload);
  if (!data.ok) {
    showNotice(shopResult, data.message || data.error || "Failed to update shop.", true);
    return;
  }
  showNotice(shopResult, "Shop updated.");
  await loadShop();
});

profileForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  profileResult.style.display = "none";
  const formData = new FormData(profileForm);
  const payload = {
    full_name: String(formData.get("full_name") || "").trim(),
    phone: String(formData.get("phone") || "").trim(),
    email: String(formData.get("email") || "").trim()
  };
  const data = await apiPost("/owner/update_profile", payload);
  if (!data.ok) {
    showNotice(profileResult, data.message || data.error || "Failed to update profile.", true);
    return;
  }
  setUser(data.user);
  ownerName.textContent = data.user.full_name || data.user.email || "Owner";
  showNotice(profileResult, "Profile updated.");
});

passwordForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  passwordResult.style.display = "none";
  const formData = new FormData(passwordForm);
  const oldPassword = String(formData.get("old_password") || "");
  const newPassword = String(formData.get("new_password") || "");
  const confirmPassword = String(formData.get("confirm_password") || "");

  if (newPassword !== confirmPassword) {
    showNotice(passwordResult, "Passwords do not match.", true);
    return;
  }

  const data = await apiPost("/auth/change_password", {
    old_password: oldPassword,
    new_password: newPassword
  });
  if (!data.ok) {
    showNotice(passwordResult, data.message || data.error || "Failed to change password.", true);
    return;
  }

  const currentUser = getUser();
  if (currentUser) {
    currentUser.must_change_password = 0;
    setUser(currentUser);
  }
  mustChangePassword = false;
  if (passwordNote) passwordNote.style.display = "none";
  showNotice(passwordResult, "Password updated. Shop activated.");
  passwordForm.reset();
  [shopForm, profileForm, createProductForm, updateProductForm].forEach((form) => {
    if (!form) return;
    Array.from(form.elements || []).forEach((el) => {
      el.disabled = false;
    });
  });
  await loadShop();
  await loadProducts();
  await loadOrders();
  setActiveTab("profile");
});

function productCard(product) {
  return `
    <div class="card">
      <div class="pill">${product.name}</div>
      <p class="muted">${product.description || "No description"}</p>
      <div><strong>Price:</strong> ${product.price}</div>
      <div><strong>Status:</strong> ${Number(product.is_active) ? "Active" : "Hidden"}</div>
      <div style="margin-top:12px; display:flex; gap:8px; flex-wrap:wrap;">
        <button class="btn secondary" data-delete="${product.id}">Delete</button>
      </div>
    </div>
  `;
}

async function loadProducts() {
  const data = await apiGet("/owner/list_products");
  if (!data.ok) {
    productsBox.innerHTML = "<div class=\"notice\">Unable to load products.</div>";
    return;
  }
  const products = data.products || [];
  productsCache = products;
  productsBox.innerHTML = products.length
    ? products.map(productCard).join("")
    : "<div class=\"notice\">No products yet.</div>";
  if (updateProductSelect) {
    updateProductSelect.innerHTML = products.length
      ? products.map((p) => `<option value="${p.id}">${p.name} (#${p.id})</option>`).join("")
      : "<option value=\"\">No products</option>";
    if (products.length) {
      updateProductSelect.value = String(products[0].id);
      updateProductSelect.dispatchEvent(new Event("change"));
    }
  }
}

updateProductSelect?.addEventListener("change", () => {
  const productId = Number(updateProductSelect.value || 0);
  const product = productsCache.find((p) => Number(p.id) === productId);
  if (!product) return;
  if (updateProductName) updateProductName.value = product.name || "";
  if (updateProductDescription) updateProductDescription.value = product.description || "";
  if (updateProductPrice) updateProductPrice.value = product.price || "";
  if (updateProductActive) updateProductActive.value = Number(product.is_active) ? "1" : "0";
});

function orderCard(order) {
  const items = (order.items || [])
    .map((item) => `${item.product_name} x${item.qty}`)
    .join(", ");
  const isCompleted = String(order.status || "") === "completed";
  return `
    <div class="card">
      <div class="pill">${order.order_code}</div>
      <p class="muted">Customer: ${order.customer_phone || "-"}</p>
      <div>Status: ${order.status}</div>
      <div>Items: ${items || "None"}</div>
      <div style="margin-top:12px; display:flex; gap:8px; flex-wrap:wrap;">
        ${isCompleted ? "" : `
          <button class="btn secondary" data-action="complete" data-id="${order.id}">
            Mark completed
          </button>
        `}
        ${isCompleted ? `
          <button class="btn secondary" data-action="review" data-id="${order.id}">
            Create review token
          </button>
        ` : ""}
      </div>
    </div>
  `;
}

async function loadOrders() {
  const data = await apiGet("/owner/list_orders");
  if (!data.ok) {
    ordersBox.innerHTML = "<div class=\"notice\">Unable to load orders.</div>";
    return;
  }
  const orders = data.orders || [];
  ordersBox.innerHTML = orders.length
    ? orders.map(orderCard).join("")
    : "<div class=\"notice\">No orders yet.</div>";
}

createProductForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  createProductResult.style.display = "none";
  const formData = new FormData(createProductForm);
  const data = await apiPostForm("/owner/create_product", formData);
  if (!data.ok) {
    showNotice(createProductResult, data.message || data.error || "Failed to create product.", true);
    return;
  }
  showNotice(createProductResult, "Product created.");
  createProductForm.reset();
  await loadProducts();
});

updateProductForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  updateProductResult.style.display = "none";
  const formData = new FormData(updateProductForm);
  const activeVal = String(formData.get("is_active") || "").trim();
  if (activeVal === "") {
    formData.delete("is_active");
  }
  const data = await apiPostForm("/owner/update_product", formData);
  if (!data.ok) {
    showNotice(updateProductResult, data.message || data.error || "Failed to update product.", true);
    return;
  }
  showNotice(updateProductResult, "Product updated.");
  updateProductForm.reset();
  await loadProducts();
});

productsBox?.addEventListener("click", async (event) => {
  const btn = event.target.closest("button[data-delete]");
  if (!btn) return;
  const productId = Number(btn.dataset.delete || 0);
  if (!productId) return;
  const data = await apiPost("/owner/delete_product", { product_id: productId });
  if (!data.ok) {
    showNotice(createProductResult, data.message || data.error || "Failed to delete product.", true);
    return;
  }
  await loadProducts();
});

ordersBox?.addEventListener("click", async (event) => {
  const btn = event.target.closest("button[data-action]");
  if (!btn) return;
  const orderId = Number(btn.dataset.id || 0);
  if (!orderId) return;

  if (btn.dataset.action === "complete") {
    const data = await apiPost("/owner/complete_order", { order_intent_id: orderId });
    if (!data.ok) {
      showNotice(orderMessage, data.message || data.error || "Failed to complete order.", true);
      return;
    }
    showNotice(orderMessage, "Order marked completed.");
    await loadOrders();
    return;
  }

  if (btn.dataset.action === "review") {
    const data = await apiPost("/owner/create_review_token", { order_intent_id: orderId });
    if (!data.ok) {
      showNotice(orderMessage, data.message || data.error || "Failed to create review token.", true);
      return;
    }
    showNotice(orderMessage, `Review link: ${data.review_url}`);
  }
});

openSecurityBtn?.addEventListener("click", () => {
  setActiveTab("security");
});

fillProfile();
if (mustChangePassword) {
  shopBox.textContent = "Change your password to access shop data.";
  productsBox.innerHTML = "<div class=\"notice\">Change your password to access products.</div>";
  ordersBox.innerHTML = "<div class=\"notice\">Change your password to access orders.</div>";
  setStat(statOrders, 0);
  setStat(statCompleted, 0);
  setStat(statReviews, 0);
} else {
  loadShop();
  loadProducts();
  loadOrders();
  loadStats();
}
